#!/bin/bash
set -euo pipefail

# ====================================================================================
# RKPi5 v2.1 Golden Master Setup
# Deterministic user/paths + verbose progress
# ====================================================================================

USER_FIXED="rk-pi5-user"
HOME_FIXED="/home/${USER_FIXED}"
DESKTOP_DIR="${HOME_FIXED}/Desktop"
SCRIPTS_DIR="${DESKTOP_DIR}/scripts"
CONTENT_DIR="${DESKTOP_DIR}/rapture_kit"
LOG_FILE="${DESKTOP_DIR}/rkpi5-setup.log"

mkdir -p "${SCRIPTS_DIR}" "${CONTENT_DIR}"
: > "${LOG_FILE}" || true


say() { echo -e "$1" | tee -a "${LOG_FILE}"; }
hr() { say "
---------------------------------------------"; }

phase_start() { PHASE_NAME="$1"; PHASE_TS_START=$(date +%s); hr; say "==> ${PHASE_NAME}"; }
phase_end() { local ts_end=$(date +%s); local dt=$((ts_end - PHASE_TS_START)); say "✅ ${PHASE_NAME} completed in ${dt}s"; }

say "🚀 RKPi5 v2.1 Golden Master Setup (Desktop‑based)"
say "User: ${USER_FIXED} | Scripts: ${SCRIPTS_DIR} | Content: ${CONTENT_DIR}"

read -r -p "📋 Proceed with setup on THIS device? (y/N): " yn; [[ $yn =~ ^[Yy]$ ]] || { say "Cancelled"; exit 1; }

# ====================================================================================
# 1) Detect USB that contains _rkpi5 and Section*
# ====================================================================================
phase_start "[1/6] Detect USB"
USB_CANDIDATES=(/media/${USER_FIXED}/*)
MATCHES=()
for m in "${USB_CANDIDATES[@]}"; do
[[ -d "$m" ]] || continue
if compgen -G "$m/_rkpi5/*.sh" > /dev/null && compgen -G "$m/Section*" > /dev/null; then
MATCHES+=("$m")
fi
done

if (( ${#MATCHES[@]} == 0 )); then
say "❌ No USB mount under /media/${USER_FIXED} with both _rkpi5/*.sh and Section*"
exit 1
fi

if (( ${#MATCHES[@]} == 1 )); then
USB_MOUNT="${MATCHES[0]}"
say "✅ USB found: ${USB_MOUNT}"
else
say "🔎 Multiple USB candidates found:"; i=1; for m in "${MATCHES[@]}"; do say " [$i] $m"; ((i++)); done
read -r -p "Select USB [1-${#MATCHES[@]}]: " choice
idx=$((choice-1))
USB_MOUNT="${MATCHES[$idx]}"
say "✅ Using: ${USB_MOUNT}"
fi
phase_end

# ====================================================================================
# 2) Copy scripts, set perms, normalize endings
# ====================================================================================
phase_start "[2/6] Stage scripts → ${SCRIPTS_DIR}"
cp -v "${USB_MOUNT}"/_rkpi5/*.sh "${SCRIPTS_DIR}/" | tee -a "${LOG_FILE}"
chmod -v +x "${SCRIPTS_DIR}"/*.sh | tee -a "${LOG_FILE}"
# Normalize endings if dos2unix exists
if command -v dos2unix >/dev/null 2>&1; then
dos2unix "${SCRIPTS_DIR}"/*.sh 2>/dev/null | tee -a "${LOG_FILE}" || true
else
say "ℹ️ dos2unix not installed yet; continuing without line-ending normalization"
fi
say "📄 Staged scripts:"; ls -l "${SCRIPTS_DIR}"/*.sh | tee -a "${LOG_FILE}"
phase_end

# ====================================================================================
# 3) Copy content (may take a while)
# ====================================================================================
phase_start "[3/6] Copy content → ${CONTENT_DIR}"
shopt -s nullglob
SECTIONS=("${USB_MOUNT}"/Section*)
if (( ${#SECTIONS[@]} == 0 )); then
say "❌ No Section* folders on USB. Aborting."; exit 1
fi
# Progress display with rsync
rsync -a --delete --info=progress2 --human-readable "${SECTIONS[@]}" "${CONTENT_DIR}/" | tee -a "${LOG_FILE}"
# Copy a top-level index if present
if [[ -f "${USB_MOUNT}/index.html" ]]; then
cp -v "${USB_MOUNT}/index.html" "${CONTENT_DIR}/index.html" | tee -a "${LOG_FILE}"
fi
# Generate minimal index if missing (prevents 403)
if [[ ! -f "${CONTENT_DIR}/index.html" ]]; then
say "🧩 Generating minimal index.html at content root to prevent 403..."
cat > "${CONTENT_DIR}/index.html" <<'HTML'
<!DOCTYPE html>
<html lang="en"><head><meta charset="utf-8"><meta name="viewport" content="width=device-width,initial-scale=1">
<title>Rapture Kit – Training Content</title>
<style>body{font-family:system-ui,Segoe UI,Arial;margin:2rem}a{display:block;margin:.25rem 0}</style>
</head><body>
<h1>Rapture Kit – Training Content</h1>
<p>Select a section folder below.</p>
</body></html>
HTML
fi
# Build sections.json if jq exists
if command -v jq >/dev/null 2>&1; then
find "${CONTENT_DIR}" -maxdepth 1 -type d -name 'Section*' | sort | \
awk -v p="/rapture_kit" '{n=$0;gsub(/^.*\//, "", n); printf("{\"name\":\"%s\",\"path\":\"%s/%s/\"}
", n, p, n)}' | \
jq -s '.' > "${CONTENT_DIR}/sections.json" || true
fi
# File/dir perms (world-read, listable)
find "${CONTENT_DIR}" -type f -exec chmod 644 {} +
find "${CONTENT_DIR}" -type d -exec chmod 755 {} +
# Summary
FILES_COUNT=$(find "${CONTENT_DIR}" -type f | wc -l || echo 0)
SIZE_HUMAN=$(du -sh "${CONTENT_DIR}" | cut -f1 || echo 0)
say "📦 Content ready: ${FILES_COUNT} files, total ${SIZE_HUMAN}"
phase_end

# ====================================================================================
# 4) Run numbered scripts in sequence (non-interactive)
# ====================================================================================
cd "${SCRIPTS_DIR}"

phase_start "[4/6] 0-check-system.sh"
./0-check-system.sh | tee -a "${LOG_FILE}"
phase_end

phase_start "[4/6] 1-install-core.sh"
./1-install-core.sh | tee -a "${LOG_FILE}"
phase_end

phase_start "[4/6] 2-configure-hotspot.sh"
./2-configure-hotspot.sh | tee -a "${LOG_FILE}"
phase_end

phase_start "[5/6] 3-configure-web.sh"
./3-configure-web.sh | tee -a "${LOG_FILE}"
phase_end

phase_start "[5/6] 4-finalize-golden-master.sh"
./4-finalize-golden-master.sh | tee -a "${LOG_FILE}"
phase_end

phase_start "[6/6] 5-validate-deployment.sh"
./5-validate-deployment.sh | tee -a "${LOG_FILE}"
phase_end

hr
say "🎉 RKPi5 v2.1 setup complete. Hotspot should be active and portal ready."
say "Log saved to: ${LOG_FILE}"
